#!/usr/bin/env python

import os
import carla
import numpy as np

from srunner.autoagents.sensor_interface import SensorInterface
from srunner.tools.route_manipulation import downsample_route

from agent.autonomous_agent import AutonomousAgent
from agent.local_planner import LocalPlanner
from comm.transceiver import Transceiver

class CoopernautReceiverAgent(AutonomousAgent):
    """
    This class is to host the coopernaut receiver agent in the environment.
    (1) It defines sensors
    (2) Generate vehicle control based on the ego observation and received representations
    """
    def __init__(self,
                 vehicle,
                 agent_config,
                 ):
        self.agent_id = agent_config.name
        self.vehicle = vehicle
        self.sensor_devices = agent_config.sensors
        # Set up the Transceiver
        self.transceiver = Transceiver(vehicle=vehicle,
                                       agent=self,
                                       sharing_mode=True,
                                      )
        # Record the history of messages
        self.message_history = []

        # Set up global plan
        self._global_plan = None
        self._global_plan_world_coord = None

        # Set up sensor interface
        self.sensor_interface = SensorInterface()
        # Set up language plan/goal/task
        self.task = agent_config.task
        # Set up the local planner
        args_lateral_dict = {
            'K_P': 1,
            'K_D': 0.4,
            'K_I': 0,
            'dt': 1.0 / 20.0}
        self.local_planner = LocalPlanner(
             self.vehicle, opt_dict={'target_speed': 20.0,
                                     'lateral_control_dict': args_lateral_dict,
                                     'max_brake': 0.75,
                                     'max_throttle': 0.75
                                     })
        self.step_count = 0

    def get_observation(self):
        """
        Get observation from sensors and messages
        """
        observation = self.sensor_interface.get_data()
        observation.update({'LIDAR': observation[f'{self.vehicle.id}_LIDAR']})
        communication_data = self.transceiver.get_repr_data()
        observation.update({'received_messages':communication_data})
        observation.update({'ego_transform':self.vehicle.get_transform()})
        observation.update({'ego_bbox':self.vehicle.bounding_box})
        observation.update({'ego_speed':self.vehicle.get_velocity().length()})
        return observation

    def run_step(self, action):
        """
        action is in the format of {'throttle': 0.0, 'steer': 0.0, 'brake': 0.0}
        """
        control = carla.VehicleControl()
        pid_control = self.local_planner.run_step()
        
        # Step 1: Get Action from the CoopernautReceiverPolicy
        control.throttle = float(action['throttle'])
        control.brake = float(action['brake'])
        control.steer = float(action['steer'])
        print("Model's Output:\t[{:.2f},\t{:.2f},\t{:.2f}]".format(control.throttle, control.brake, control.steer))
        
        # Step 2: Wrap Control to be safe
        if control.brake >= control.throttle or control.brake>0.7:
            control.throttle = 0.0
        else:
            control.brake = 0.0
        if pid_control.brake > 0:
            control.throttle = 0.0
            control.brake = pid_control.brake
        control.throttle = np.clip(control.throttle,0.0,0.75)
        control.brake = np.clip(control.brake,0.0,0.75)
        control.steer = np.clip(control.steer,-1.0,1.0)
        print("Final Control:\t[{:.2f},\t{:.2f},\t{:.2f}]".format(control.throttle, control.brake, control.steer))
        self.step_count += 1
        return control

    def set_global_plan(self, global_plan_gps, global_plan_world_coord):
        """
        Set global plan (route) for the agent
        """
        ds_ids = downsample_route(global_plan_world_coord, 1)
        self._global_plan_world_coord = [(global_plan_world_coord[x][0], global_plan_world_coord[x][1])
                                         for x in ds_ids]
        self._global_plan = [global_plan_gps[x] for x in ds_ids]

    def sensors(self):  # pylint: disable=no-self-use
        """
        :return: a list of sensors attached to the agent
        """
        sensors = []
        if 'Center' in self.sensor_devices:
            sensor_spec = self.sensor_devices['Center']
            sensors.append(
            {'type': 'sensor.camera.rgb', 'x': 0.7, 'y': 0.0, 'z': 1.60, 'roll': 0.0, 'pitch': 0.0, 'yaw': 0.0,
             'width': float(sensor_spec.camera_width), 'height': float(sensor_spec.camera_height), 'fov': 100, 'id': 'Center'},
            )
        if 'LIDAR' in self.sensor_devices:
            sensor_spec = self.sensor_devices['LIDAR']
            sensors.append(
            {'type': 'sensor.lidar.ray_cast', 
             'x': 0.0, 'y': 0.0,'z': 0.5,  
             'yaw': 90.0, 'pitch': 0.0, 'roll': 0.0,
             'range': 50.0,
             'rotation_frequency': 20, 
             'channels': 64,
             'upper_fov': 4, 
             'lower_fov': -20, 
             'points_per_second': 2304000,
             'id': f'{self.vehicle.id}_LIDAR'},
            )
        return sensors
